<?php

if (!defined('ABSPATH')) {
    exit;
}

class Lokalisa_Settings
{
    private string $option_name = 'lokalisa_settings';

    public function register(): void
    {
        add_action('admin_menu', [$this, 'register_menu']);
        add_action('admin_init', [$this, 'register_settings']);
    }

    public function get_option(string $key, $default = '')
    {
        $options = get_option($this->option_name, []);
        return $options[$key] ?? $default;
    }

    public function get_all(): array
    {
        $options = get_option($this->option_name, []);
        return is_array($options) ? $options : [];
    }

    public function register_menu(): void
    {
        add_submenu_page(
            'woocommerce',
            __('Lokalisa', 'lokalisa-entregas-locales'),
            __('Lokalisa', 'lokalisa-entregas-locales'),
            'manage_woocommerce',
            'lokalisa-settings',
            [$this, 'render_page']
        );
    }

    public function register_settings(): void
    {
        register_setting('lokalisa_settings_group', $this->option_name, [
            'type' => 'array',
            'sanitize_callback' => [$this, 'sanitize'],
        ]);

        add_settings_section('lokalisa_main', __('Configuracion API', 'lokalisa-entregas-locales'), '__return_false', 'lokalisa-settings');

        add_settings_field('base_url', __('Base URL', 'lokalisa-entregas-locales'), [$this, 'field_base_url'], 'lokalisa-settings', 'lokalisa_main');
        add_settings_field('api_key', __('API Key', 'lokalisa-entregas-locales'), [$this, 'field_api_key'], 'lokalisa-settings', 'lokalisa_main');
        add_settings_field('api_secret', __('API Secret', 'lokalisa-entregas-locales'), [$this, 'field_api_secret'], 'lokalisa-settings', 'lokalisa_main');
        add_settings_field('pickup_address_id', __('Pickup Address', 'lokalisa-entregas-locales'), [$this, 'field_pickup_address'], 'lokalisa-settings', 'lokalisa_main');
        add_settings_field('shipping_title', __('Titulo de envio', 'lokalisa-entregas-locales'), [$this, 'field_shipping_title'], 'lokalisa-settings', 'lokalisa_main');
        add_settings_field('geocode_provider', __('Proveedor de ubicacion', 'lokalisa-entregas-locales'), [$this, 'field_geocode_provider'], 'lokalisa-settings', 'lokalisa_main');
        add_settings_field('google_api_key', __('Google API Key', 'lokalisa-entregas-locales'), [$this, 'field_google_api_key'], 'lokalisa-settings', 'lokalisa_main');
        add_settings_field('debug', __('Debug', 'lokalisa-entregas-locales'), [$this, 'field_debug'], 'lokalisa-settings', 'lokalisa_main');
    }

    public function sanitize(array $input): array
    {
        $provider = (string) ($input['geocode_provider'] ?? 'osm');
        if (!in_array($provider, ['osm', 'google'], true)) {
            $provider = 'osm';
        }

        return [
            'base_url' => esc_url_raw(trim((string) ($input['base_url'] ?? ''))),
            'api_key' => sanitize_text_field((string) ($input['api_key'] ?? '')),
            'api_secret' => sanitize_text_field((string) ($input['api_secret'] ?? '')),
            'pickup_address_id' => (int) ($input['pickup_address_id'] ?? 0),
            'shipping_title' => sanitize_text_field((string) ($input['shipping_title'] ?? 'Lokalisa - entregas locales')),
            'geocode_provider' => $provider,
            'google_api_key' => sanitize_text_field((string) ($input['google_api_key'] ?? '')),
            'debug' => !empty($input['debug']) ? '1' : '0',
        ];
    }

    public function render_page(): void
    {
        echo '<div class="wrap">';
        echo '<h1>' . esc_html__('Lokalisa - entregas locales', 'lokalisa-entregas-locales') . '</h1>';
        echo '<form method="post" action="options.php">';
        settings_fields('lokalisa_settings_group');
        do_settings_sections('lokalisa-settings');
        submit_button();
        echo '</form>';

        echo '<hr />';
        echo '<h2>' . esc_html__('Sincronizacion manual', 'lokalisa-entregas-locales') . '</h2>';
        if (!empty($_GET['lokalisa_sync_message'])) {
            $status = sanitize_text_field((string) ($_GET['lokalisa_sync'] ?? ''));
            $class = $status === 'ok' ? 'updated' : 'error';
            echo '<div class="notice ' . esc_attr($class) . '"><p>' . esc_html(wp_unslash($_GET['lokalisa_sync_message'])) . '</p></div>';
        }
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
        echo '<input type="hidden" name="action" value="lokalisa_sync_catalog" />';
        wp_nonce_field('lokalisa_sync_catalog');
        submit_button(__('Sincronizar catalogo y precios', 'lokalisa-entregas-locales'), 'secondary');
        echo '</form>';
        echo '</div>';
    }

    public function field_base_url(): void
    {
        $value = esc_attr($this->get_option('base_url', 'https://sandbox-api.lokalisa.com'));
        echo '<input type="url" name="' . esc_attr($this->option_name) . '[base_url]" value="' . $value . '" class="regular-text" />';
        echo '<p class="description">Ej: https://sandbox-api.lokalisa.com o https://api.lokalisa.com</p>';
    }

    public function field_api_key(): void
    {
        $value = esc_attr($this->get_option('api_key', ''));
        echo '<input type="text" name="' . esc_attr($this->option_name) . '[api_key]" value="' . $value . '" class="regular-text" />';
    }

    public function field_api_secret(): void
    {
        $value = esc_attr($this->get_option('api_secret', ''));
        echo '<input type="password" name="' . esc_attr($this->option_name) . '[api_secret]" value="' . $value . '" class="regular-text" autocomplete="new-password" />';
    }

    public function field_pickup_address(): void
    {
        $selected = (int) $this->get_option('pickup_address_id', 0);
        $options = apply_filters('lokalisa_pickup_addresses', []);

        echo '<select name="' . esc_attr($this->option_name) . '[pickup_address_id]">';
        echo '<option value="0">' . esc_html__('Selecciona un pickup', 'lokalisa-entregas-locales') . '</option>';
        foreach ($options as $address) {
            $id = (int) ($address['id'] ?? 0);
            $label = (string) ($address['label'] ?? '');
            if ($id <= 0 || $label === '') {
                continue;
            }
            printf('<option value="%d" %s>%s</option>', $id, selected($selected, $id, false), esc_html($label));
        }
        echo '</select>';
        echo '<p class="description">Sincroniza addresses desde la API (se cargan al abrir esta pagina).</p>';
    }

    public function field_shipping_title(): void
    {
        $value = esc_attr($this->get_option('shipping_title', 'Lokalisa - entregas locales'));
        echo '<input type="text" name="' . esc_attr($this->option_name) . '[shipping_title]" value="' . $value . '" class="regular-text" />';
    }

    public function field_geocode_provider(): void
    {
        $value = (string) $this->get_option('geocode_provider', 'osm');
        echo '<select name="' . esc_attr($this->option_name) . '[geocode_provider]">';
        echo '<option value="osm" ' . selected($value, 'osm', false) . '>OpenStreetMap (Nominatim)</option>';
        echo '<option value="google" ' . selected($value, 'google', false) . '>Google Places</option>';
        echo '</select>';
        echo '<p class="description">Selecciona el proveedor para autocompletar direcciones y lat/lng.</p>';
    }

    public function field_google_api_key(): void
    {
        $value = esc_attr($this->get_option('google_api_key', ''));
        echo '<input type="text" name="' . esc_attr($this->option_name) . '[google_api_key]" value="' . $value . '" class="regular-text" />';
        echo '<p class="description">Solo requerido si usas Google Places.</p>';
    }

    public function field_debug(): void
    {
        $checked = $this->get_option('debug', '0') === '1';
        echo '<label><input type="checkbox" name="' . esc_attr($this->option_name) . '[debug]" value="1" ' . checked($checked, true, false) . ' /> ';
        echo esc_html__('Registrar logs basicos en debug.log', 'lokalisa-entregas-locales') . '</label>';
    }
}
