<?php

if (!defined('ABSPATH')) {
    exit;
}

class Lokalisa_Checkout
{
    private Lokalisa_Api_Client $api;
    private Lokalisa_Settings $settings;

    public function __construct(Lokalisa_Api_Client $api, Lokalisa_Settings $settings)
    {
        $this->api = $api;
        $this->settings = $settings;
    }

    public function register(): void
    {
        add_filter('woocommerce_checkout_fields', [$this, 'add_fields']);
        add_action('woocommerce_checkout_update_order_review', [$this, 'capture_checkout_fields']);
        add_action('woocommerce_checkout_process', [$this, 'validate_fields']);
        add_action('woocommerce_checkout_create_order', [$this, 'store_order_meta'], 10, 2);
        add_action('woocommerce_checkout_order_processed', [$this, 'create_lokalisa_order'], 20, 3);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    public function enqueue_assets(): void
    {
        if (!function_exists('is_checkout') || !is_checkout()) {
            return;
        }

        wp_enqueue_script('jquery-ui-datepicker');
        wp_enqueue_style('lokalisa-checkout', LOKALISA_PLUGIN_URL . 'assets/css/checkout.css', [], '0.1.0');
        wp_enqueue_script('lokalisa-checkout', LOKALISA_PLUGIN_URL . 'assets/js/checkout.js', ['jquery', 'jquery-ui-datepicker'], '0.1.0', true);

        $provider = (string) $this->settings->get_option('geocode_provider', 'osm');
        $googleKey = (string) $this->settings->get_option('google_api_key', '');
        $deliveryDefault = $this->next_delivery_date();

        wp_localize_script('lokalisa-checkout', 'lokalisaCheckout', [
            'provider' => $provider,
            'googleKey' => $googleKey,
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('lokalisa_geocode'),
            'minDate' => $deliveryDefault,
            'mapCenter' => [
                'lat' => 19.4326,
                'lng' => -99.1332,
            ],
        ]);

        if ($provider === 'google' && $googleKey !== '') {
            wp_enqueue_script(
                'lokalisa-google-places',
                'https://maps.googleapis.com/maps/api/js?key=' . rawurlencode($googleKey) . '&libraries=places',
                [],
                null,
                true
            );
        }

        if ($provider === 'osm') {
            wp_enqueue_style('lokalisa-leaflet', 'https://unpkg.com/leaflet@1.9.4/dist/leaflet.css', [], '1.9.4');
            wp_enqueue_script('lokalisa-leaflet', 'https://unpkg.com/leaflet@1.9.4/dist/leaflet.js', [], '1.9.4', true);
        }
    }

    public function add_fields(array $fields): array
    {
        $fields['shipping']['lokalisa_address_search'] = [
            'type' => 'text',
            'label' => 'Buscar direccion',
            'required' => false,
            'class' => ['form-row-wide'],
            'priority' => 118,
        ];
        $fields['shipping']['lokalisa_delivery_date'] = [
            'type' => 'text',
            'label' => 'Fecha de entrega',
            'required' => false,
            'class' => ['form-row-first'],
            'priority' => 119,
        ];
        $fields['shipping']['lokalisa_dropoff_lat'] = [
            'type' => 'text',
            'label' => 'Latitud de entrega',
            'required' => false,
            'class' => ['form-row-first'],
            'priority' => 121,
        ];
        $fields['shipping']['lokalisa_dropoff_lng'] = [
            'type' => 'text',
            'label' => 'Longitud de entrega',
            'required' => false,
            'class' => ['form-row-last'],
            'priority' => 122,
        ];

        return $fields;
    }

    public function capture_checkout_fields(string $posted): void
    {
        parse_str($posted, $data);
        $lat = $this->get_post_value($data, 'lokalisa_dropoff_lat');
        $lng = $this->get_post_value($data, 'lokalisa_dropoff_lng');
        $date = $this->get_post_value($data, 'lokalisa_delivery_date');

        if (WC()->session) {
            WC()->session->set('lokalisa_dropoff_lat', $lat);
            WC()->session->set('lokalisa_dropoff_lng', $lng);
            WC()->session->set('lokalisa_delivery_date', $date);
        }
    }

    public function validate_fields(): void
    {
        if (!$this->is_lokalisa_selected()) {
            return;
        }

        $lat = $this->get_post_value($_POST, 'lokalisa_dropoff_lat');
        $lng = $this->get_post_value($_POST, 'lokalisa_dropoff_lng');
        $date = $this->get_post_value($_POST, 'lokalisa_delivery_date');

        if ($lat === '' || $lng === '') {
            wc_add_notice(__('Por favor ingresa latitud y longitud para la entrega.', 'lokalisa-entregas-locales'), 'error');
        }

        if ($date !== '' && !$this->is_valid_delivery_date($date)) {
            wc_add_notice(__('La fecha de entrega no es valida. Selecciona un dia habil.', 'lokalisa-entregas-locales'), 'error');
        }
    }

    public function store_order_meta($order, array $data): void
    {
        if (!($order instanceof WC_Order)) {
            return;
        }

        $lat = $this->get_post_value($_POST, 'lokalisa_dropoff_lat');
        $lng = $this->get_post_value($_POST, 'lokalisa_dropoff_lng');
        $date = $this->get_post_value($_POST, 'lokalisa_delivery_date');

        if ($lat !== '') {
            $order->update_meta_data('_lokalisa_dropoff_lat', $lat);
        }
        if ($lng !== '') {
            $order->update_meta_data('_lokalisa_dropoff_lng', $lng);
        }
        if ($date !== '') {
            $order->update_meta_data('_lokalisa_delivery_date', $date);
        }
    }

    public function create_lokalisa_order(int $orderId, array $postedData, $order): void
    {
        if (!($order instanceof WC_Order)) {
            return;
        }

        if (!$this->order_uses_lokalisa($order)) {
            return;
        }

        if ($order->get_meta('_lokalisa_order_id')) {
            return;
        }

        $pickupId = (int) $this->settings->get_option('pickup_address_id', 0);
        if ($pickupId <= 0) {
            return;
        }

        $lat = (string) $order->get_meta('_lokalisa_dropoff_lat');
        $lng = (string) $order->get_meta('_lokalisa_dropoff_lng');
        $date = (string) $order->get_meta('_lokalisa_delivery_date');

        $payload = [
            'external_id' => 'WC-' . $orderId,
            'pickup_address_id' => $pickupId,
            'dropoff_name' => $this->resolve_dropoff_name($order),
            'dropoff_phone' => $this->resolve_dropoff_phone($order),
            'dropoff_address' => $this->resolve_dropoff_address($order),
            'dropoff_lat' => $lat !== '' ? (float) $lat : null,
            'dropoff_lng' => $lng !== '' ? (float) $lng : null,
            'delivery_date' => $this->normalize_delivery_date($date),
            'payment_mode' => $this->resolve_payment_mode($order),
            'cod_amount' => $this->resolve_cod_amount($order),
            'cod_includes_shipping' => true,
            'items' => $this->build_items($order),
        ];

        $idempotencyKey = 'wc-order-' . $orderId;
        $response = $this->api->create_order($payload, $idempotencyKey);

        if (is_array($response) && !empty($response['ok'])) {
            $order->update_meta_data('_lokalisa_order_id', $response['order']['id'] ?? null);
            $order->update_meta_data('_lokalisa_public_code', $response['order']['public_code'] ?? null);
            $order->update_meta_data('_lokalisa_tracking_url', $response['order']['tracking_url'] ?? null);
            $order->save();
        }
    }

    private function build_items(WC_Order $order): array
    {
        $items = [];
        foreach ($order->get_items('line_item') as $item) {
            $product = $item->get_product();
            $unitPrice = $item->get_total() / max(1, $item->get_quantity());

            $items[] = [
                'name' => $item->get_name(),
                'description' => $product ? $product->get_short_description() : null,
                'quantity' => (int) $item->get_quantity(),
                'unit_price' => (float) $unitPrice,
            ];
        }

        return $items;
    }

    private function resolve_payment_mode(WC_Order $order): string
    {
        if ($order->is_paid()) {
            return 'PAID';
        }

        if ($order->get_payment_method() === 'cod') {
            return 'COD';
        }

        return 'ONLINE';
    }

    private function resolve_cod_amount(WC_Order $order): ?float
    {
        if ($order->get_payment_method() !== 'cod') {
            return null;
        }

        return (float) $order->get_total();
    }

    private function resolve_dropoff_name(WC_Order $order): string
    {
        $name = trim($order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name());
        if ($name !== '') {
            return $name;
        }

        return trim($order->get_billing_first_name() . ' ' . $order->get_billing_last_name());
    }

    private function resolve_dropoff_phone(WC_Order $order): string
    {
        $phone = (string) $order->get_billing_phone();
        if ($phone !== '') {
            return $phone;
        }

        return (string) $order->get_shipping_phone();
    }

    private function resolve_dropoff_address(WC_Order $order): string
    {
        $line1 = trim((string) $order->get_shipping_address_1());
        $line2 = trim((string) $order->get_shipping_address_2());
        $city = trim((string) $order->get_shipping_city());

        if ($line1 === '' && $line2 === '' && $city === '') {
            $line1 = trim((string) $order->get_billing_address_1());
            $line2 = trim((string) $order->get_billing_address_2());
            $city = trim((string) $order->get_billing_city());
        }

        $parts = array_filter([$line1, $line2, $city]);
        return implode(', ', $parts);
    }

    private function order_uses_lokalisa(WC_Order $order): bool
    {
        foreach ($order->get_shipping_methods() as $method) {
            if ($method->get_method_id() === 'lokalisa') {
                return true;
            }
        }

        return false;
    }

    private function is_lokalisa_selected(): bool
    {
        $chosen = WC()->session ? WC()->session->get('chosen_shipping_methods') : [];
        if (!is_array($chosen)) {
            return false;
        }

        foreach ($chosen as $method) {
            if (str_starts_with((string) $method, 'lokalisa')) {
                return true;
            }
        }

        return false;
    }

    private function is_valid_delivery_date(string $date): bool
    {
        $timestamp = strtotime($date);
        if (!$timestamp) {
            return false;
        }

        $min = strtotime($this->next_delivery_date());
        if ($timestamp < $min) {
            return false;
        }

        return (int) gmdate('w', $timestamp) !== 0;
    }

    private function normalize_delivery_date(string $date): string
    {
        if ($date !== '' && $this->is_valid_delivery_date($date)) {
            return gmdate('Y-m-d', strtotime($date));
        }

        return $this->next_delivery_date();
    }

    private function next_delivery_date(): string
    {
        $date = gmdate('Y-m-d', strtotime('+1 day'));
        while ((int) gmdate('w', strtotime($date)) === 0) {
            $date = gmdate('Y-m-d', strtotime($date . ' +1 day'));
        }

        return $date;
    }

    private function get_post_value(array $data, string $key): string
    {
        if (isset($data[$key])) {
            return (string) $data[$key];
        }

        $prefixed = 'shipping_' . $key;
        if (isset($data[$prefixed])) {
            return (string) $data[$prefixed];
        }

        return '';
    }
}
