<?php

if (!defined('ABSPATH')) {
    exit;
}

class Lokalisa_Catalog_Sync
{
    private Lokalisa_Api_Client $api;
    private Lokalisa_Settings $settings;

    public function __construct(Lokalisa_Api_Client $api, Lokalisa_Settings $settings)
    {
        $this->api = $api;
        $this->settings = $settings;
    }

    public function register(): void
    {
        add_action('admin_post_lokalisa_sync_catalog', [$this, 'handle_sync']);
    }

    public function handle_sync(): void
    {
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Forbidden');
        }

        check_admin_referer('lokalisa_sync_catalog');

        $result = $this->sync_prices();
        $redirect = add_query_arg([
            'page' => 'lokalisa-settings',
            'lokalisa_sync' => $result['status'],
            'lokalisa_sync_message' => rawurlencode($result['message']),
        ], admin_url('admin.php'));

        wp_safe_redirect($redirect);
        exit;
    }

    private function sync_prices(): array
    {
        $products = $this->api->get_products();
        if (empty($products)) {
            return [
                'status' => 'error',
                'message' => 'No se pudo obtener el catalogo de Lokalisa.',
            ];
        }

        $updated = 0;
        $created = 0;
        $skipped = 0;

        foreach ($products as $remote) {
            $sku = trim((string) ($remote['sku'] ?? ''));
            if ($sku === '') {
                $skipped++;
                continue;
            }

            $productId = wc_get_product_id_by_sku($sku);
            $product = $productId ? wc_get_product($productId) : null;

            if (!$product) {
                $product = new WC_Product_Simple();
                $product->set_name((string) ($remote['name'] ?? $sku));
                $product->set_status('publish');
                $product->set_sku($sku);
                $created++;
            }

            $price = (float) ($remote['price'] ?? 0);
            $product->set_regular_price((string) $price);
            $product->set_price((string) $price);

            if (!empty($remote['description'])) {
                $product->set_short_description((string) $remote['description']);
            }

            if (!empty($remote['manage_stock'])) {
                $product->set_manage_stock(true);
                $product->set_stock_quantity((int) ($remote['available_stock'] ?? 0));
                $product->set_stock_status(((int) ($remote['available_stock'] ?? 0)) > 0 ? 'instock' : 'outofstock');
            }

            $product->save();
            $updated++;
        }

        return [
            'status' => 'ok',
            'message' => sprintf('Sincronizacion completa. Actualizados: %d. Creados: %d. Omitidos: %d.', $updated, $created, $skipped),
        ];
    }
}
