<?php

if (!defined('ABSPATH')) {
    exit;
}

class Lokalisa_Api_Client
{
    private Lokalisa_Settings $settings;

    public function __construct(Lokalisa_Settings $settings)
    {
        $this->settings = $settings;
        add_filter('lokalisa_pickup_addresses', [$this, 'load_pickup_addresses']);
    }

    public function load_pickup_addresses(array $current): array
    {
        $addresses = $this->get_addresses();
        if (empty($addresses)) {
            return $current;
        }

        $options = [];
        foreach ($addresses as $address) {
            $label = trim((string) ($address['name'] ?? ''));
            $line = trim((string) ($address['address_line'] ?? ''));
            $city = trim((string) ($address['city'] ?? ''));
            $parts = array_filter([$label, $line, $city]);
            $options[] = [
                'id' => (int) ($address['id'] ?? 0),
                'label' => implode(' - ', $parts),
            ];
        }

        return $options;
    }

    public function get_addresses(): array
    {
        $response = $this->request('GET', '/api/v1/catalog/addresses');
        if (!$response || empty($response['ok'])) {
            return [];
        }

        return $response['addresses'] ?? [];
    }

    public function get_products(): array
    {
        $response = $this->request('GET', '/api/v1/catalog/products');
        if (!$response || empty($response['ok'])) {
            return [];
        }

        return $response['products'] ?? [];
    }

    public function quote(array $payload): ?array
    {
        $response = $this->request('POST', '/api/v1/quotes', $payload);
        if (!$response || empty($response['ok'])) {
            return null;
        }

        return $response['quote'] ?? null;
    }

    public function create_order(array $payload, string $idempotencyKey): ?array
    {
        $headers = [
            'Idempotency-Key' => $idempotencyKey,
        ];

        $response = $this->request('POST', '/api/v1/orders', $payload, $headers);
        if (!$response || empty($response['ok'])) {
            return $response;
        }

        return $response;
    }

    private function request(string $method, string $path, array $body = [], array $headers = []): ?array
    {
        $baseUrl = rtrim((string) $this->settings->get_option('base_url', ''), '/');
        if ($baseUrl === '') {
            return null;
        }

        $token = $this->get_access_token();
        if (!$token) {
            return null;
        }

        $url = $baseUrl . $path;
        $args = [
            'method' => $method,
            'timeout' => 20,
            'headers' => array_merge([
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $token,
            ], $headers),
        ];

        if ($method !== 'GET') {
            $args['headers']['Content-Type'] = 'application/json';
            $args['body'] = wp_json_encode($body);
        }

        $response = wp_remote_request($url, $args);
        if (is_wp_error($response)) {
            $this->log('request_error', $response->get_error_message());
            return null;
        }

        $status = wp_remote_retrieve_response_code($response);
        $raw = wp_remote_retrieve_body($response);
        $data = json_decode($raw, true);

        if ($status === 401) {
            $this->invalidate_token();
        }

        if (!is_array($data)) {
            $this->log('invalid_response', $raw);
            return null;
        }

        return $data;
    }

    private function get_access_token(): ?string
    {
        $token = get_transient('lokalisa_access_token');
        $expiresAt = (int) get_transient('lokalisa_access_token_expires_at');

        if ($token && $expiresAt > time()) {
            return (string) $token;
        }

        $tokenResponse = $this->fetch_token();
        if (!$tokenResponse) {
            return null;
        }

        $accessToken = $tokenResponse['access_token'] ?? null;
        $expiresIn = (int) ($tokenResponse['expires_in'] ?? 0);

        if (!$accessToken || $expiresIn <= 0) {
            return null;
        }

        $ttl = max(60, $expiresIn - 60);
        set_transient('lokalisa_access_token', $accessToken, $ttl);
        set_transient('lokalisa_access_token_expires_at', time() + $ttl, $ttl);

        return (string) $accessToken;
    }

    private function fetch_token(): ?array
    {
        $baseUrl = rtrim((string) $this->settings->get_option('base_url', ''), '/');
        $apiKey = (string) $this->settings->get_option('api_key', '');
        $apiSecret = (string) $this->settings->get_option('api_secret', '');

        if ($baseUrl === '' || $apiKey === '' || $apiSecret === '') {
            return null;
        }

        $url = $baseUrl . '/api/v1/auth/token';
        $response = wp_remote_post($url, [
            'timeout' => 20,
            'headers' => [
                'Accept' => 'application/json',
                'X-API-Key' => $apiKey,
                'X-API-Secret' => $apiSecret,
            ],
        ]);

        if (is_wp_error($response)) {
            $this->log('token_error', $response->get_error_message());
            return null;
        }

        $raw = wp_remote_retrieve_body($response);
        $data = json_decode($raw, true);

        if (!is_array($data)) {
            $this->log('token_invalid', $raw);
            return null;
        }

        if (empty($data['ok'])) {
            $this->log('token_rejected', $data);
            return null;
        }

        return $data;
    }

    private function invalidate_token(): void
    {
        delete_transient('lokalisa_access_token');
        delete_transient('lokalisa_access_token_expires_at');
    }

    private function log(string $code, $context): void
    {
        $debug = $this->settings->get_option('debug', '0') === '1';
        if (!$debug) {
            return;
        }

        $message = '[lokalisa] ' . $code . ' ' . wp_json_encode($context);
        error_log($message);
    }
}
