<?php

if (!defined('ABSPATH')) {
    exit;
}

class Lokalisa_Ajax
{
    private Lokalisa_Settings $settings;

    public function __construct(Lokalisa_Settings $settings)
    {
        $this->settings = $settings;
    }

    public function register(): void
    {
        add_action('wp_ajax_lokalisa_geocode_search', [$this, 'geocode_search']);
        add_action('wp_ajax_nopriv_lokalisa_geocode_search', [$this, 'geocode_search']);
    }

    public function geocode_search(): void
    {
        check_ajax_referer('lokalisa_geocode');

        $query = trim((string) ($_GET['q'] ?? ''));
        if ($query === '') {
            wp_send_json_error(['message' => 'Query required'], 400);
        }

        $provider = (string) $this->settings->get_option('geocode_provider', 'osm');
        if ($provider !== 'osm') {
            wp_send_json_error(['message' => 'Provider not supported'], 422);
        }

        $url = 'https://nominatim.openstreetmap.org/search';
        $params = [
            'format' => 'json',
            'limit' => 5,
            'addressdetails' => 1,
            'q' => $query,
        ];

        $response = wp_remote_get(add_query_arg($params, $url), [
            'timeout' => 10,
            'headers' => [
                'Accept' => 'application/json',
                'User-Agent' => 'LokalisaWoo/0.1 (support@lokalisa.com)',
            ],
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => $response->get_error_message()], 500);
        }

        $raw = wp_remote_retrieve_body($response);
        $data = json_decode($raw, true);

        if (!is_array($data)) {
            wp_send_json_error(['message' => 'Invalid response'], 500);
        }

        $items = [];
        foreach ($data as $item) {
            if (!isset($item['lat'], $item['lon'])) {
                continue;
            }
            $address = $item['address'] ?? [];

            $items[] = [
                'label' => (string) ($item['display_name'] ?? ''),
                'lat' => (float) $item['lat'],
                'lng' => (float) $item['lon'],
                'address' => [
                    'road' => (string) ($address['road'] ?? ''),
                    'house_number' => (string) ($address['house_number'] ?? ''),
                    'city' => (string) ($address['city'] ?? $address['town'] ?? $address['village'] ?? ''),
                    'state' => (string) ($address['state'] ?? ''),
                    'postcode' => (string) ($address['postcode'] ?? ''),
                    'country_code' => strtoupper((string) ($address['country_code'] ?? '')),
                ],
            ];
        }

        wp_send_json_success(['results' => $items]);
    }
}
