(function ($) {
  var mapInstance = null;
  var mapMarker = null;
  var lastSearchQuery = '';

  function fieldSelector(name) {
    return 'input[name="' + name + '"], input[name="shipping_' + name + '"]';
  }

  function setField(name, value) {
    var $field = $(fieldSelector(name));
    if ($field.length) {
      $field.val(value);
    }
  }

  function getField(name) {
    var $field = $(fieldSelector(name));
    if ($field.length) {
      return $field.val();
    }
    return '';
  }

  function buildAddressQuery() {
    var parts = [
      getField('shipping_address_1'),
      getField('shipping_address_2'),
      getField('shipping_city'),
      getField('shipping_state'),
      getField('shipping_postcode'),
      getField('shipping_country')
    ].filter(function (v) {
      return v && v.toString().trim() !== '';
    });

    return parts.join(', ');
  }

  function initDatepicker() {
    var $date = $(fieldSelector('lokalisa_delivery_date'));
    if (!$date.length) return;

    var minDate = lokalisaCheckout && lokalisaCheckout.minDate ? lokalisaCheckout.minDate : null;
    $date.datepicker({
      dateFormat: 'yy-mm-dd',
      minDate: minDate,
      beforeShowDay: function (date) {
        var day = date.getDay();
        return [day !== 0, ''];
      }
    });

    if (!$date.val() && minDate) {
      $date.val(minDate);
    }
  }

  function setLatLng(lat, lng) {
    setField('lokalisa_dropoff_lat', lat);
    setField('lokalisa_dropoff_lng', lng);
    updateMap(lat, lng);
    $(document.body).trigger('update_checkout');
  }

  function updateMap(lat, lng) {
    if (!mapInstance) return;
    var position = [parseFloat(lat), parseFloat(lng)];
    if (isNaN(position[0]) || isNaN(position[1])) return;

    if (mapInstance.setView) {
      mapInstance.setView(position, 15);
      if (!mapMarker) {
        mapMarker = L.marker(position).addTo(mapInstance);
      } else {
        mapMarker.setLatLng(position);
      }
      return;
    }

    if (mapInstance instanceof google.maps.Map) {
      var gpos = { lat: position[0], lng: position[1] };
      mapInstance.setCenter(gpos);
      mapInstance.setZoom(15);
      if (!mapMarker) {
        mapMarker = new google.maps.Marker({ position: gpos, map: mapInstance });
      } else {
        mapMarker.setPosition(gpos);
      }
    }
  }

  function initMapContainer() {
    var $search = $(fieldSelector('lokalisa_address_search'));
    if (!$search.length) return;
    if ($('#lokalisa-map').length) return;

    var $map = $('<div id="lokalisa-map" class="lokalisa-map"></div>');
    $search.closest('.form-row').after($map);
  }

  function initLeafletMap() {
    if (typeof L === 'undefined') return;
    initMapContainer();

    var center = lokalisaCheckout && lokalisaCheckout.mapCenter ? lokalisaCheckout.mapCenter : { lat: 19.4326, lng: -99.1332 };
    mapInstance = L.map('lokalisa-map').setView([center.lat, center.lng], 12);
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
      maxZoom: 19,
      attribution: '© OpenStreetMap'
    }).addTo(mapInstance);

    mapInstance.on('click', function (e) {
      setLatLng(e.latlng.lat, e.latlng.lng);
    });
  }

  function initGoogleMap() {
    initMapContainer();

    var center = lokalisaCheckout && lokalisaCheckout.mapCenter ? lokalisaCheckout.mapCenter : { lat: 19.4326, lng: -99.1332 };
    mapInstance = new google.maps.Map(document.getElementById('lokalisa-map'), {
      center: { lat: center.lat, lng: center.lng },
      zoom: 12
    });

    mapInstance.addListener('click', function (e) {
      setLatLng(e.latLng.lat(), e.latLng.lng());
    });
  }

  function fillAddressFields(data) {
    if (!data) return;

    if (data.address1) setField('shipping_address_1', data.address1);
    if (data.address2) setField('shipping_address_2', data.address2);
    if (data.city) setField('shipping_city', data.city);
    if (data.postcode) setField('shipping_postcode', data.postcode);
    if (data.state) setField('shipping_state', data.state);
    if (data.country) setField('shipping_country', data.country);
  }

  function initOsmSearch() {
    var $search = $(fieldSelector('lokalisa_address_search'));
    if (!$search.length) return;

    var $list = $('<div class="lokalisa-suggestions"></div>');
    $search.after($list);

    var timer = null;

    $search.on('input', function () {
      var query = $search.val();
      if (timer) clearTimeout(timer);
      if (!query || query.length < 3) {
        $list.empty().hide();
        return;
      }

      timer = setTimeout(function () {
        $.getJSON(lokalisaCheckout.ajaxUrl, {
          action: 'lokalisa_geocode_search',
          q: query,
          _ajax_nonce: lokalisaCheckout.nonce
        }).done(function (resp) {
          $list.empty();
          if (!resp || !resp.success || !resp.data || !resp.data.results) {
            $list.hide();
            return;
          }
          resp.data.results.forEach(function (item) {
            var $item = $('<div class="lokalisa-suggestion"></div>');
            $item.text(item.label || '');
            $item.data('lat', item.lat);
            $item.data('lng', item.lng);
            $item.data('address', item.address || {});
            $list.append($item);
          });
          $list.show();
        }).fail(function () {
          $list.hide();
        });
      }, 300);
    });

    $list.on('click', '.lokalisa-suggestion', function () {
      var $item = $(this);
      var label = $item.text();
      var lat = $item.data('lat');
      var lng = $item.data('lng');
      var addr = $item.data('address') || {};
      $search.val(label);
      setLatLng(lat, lng);

      var address1 = [addr.road, addr.house_number].filter(Boolean).join(' ');
      fillAddressFields({
        address1: address1,
        city: addr.city || '',
        postcode: addr.postcode || '',
        state: addr.state || '',
        country: addr.country_code || ''
      });

      $list.empty().hide();
    });

    $(document).on('click', function (e) {
      if (!$(e.target).closest($search).length && !$(e.target).closest($list).length) {
        $list.hide();
      }
    });
  }

  function osmGeocodeAddress(query) {
    if (!query || query.length < 3) return;
    if (query === lastSearchQuery) return;
    lastSearchQuery = query;

    $.getJSON(lokalisaCheckout.ajaxUrl, {
      action: 'lokalisa_geocode_search',
      q: query,
      _ajax_nonce: lokalisaCheckout.nonce
    }).done(function (resp) {
      if (!resp || !resp.success || !resp.data || !resp.data.results || !resp.data.results.length) {
        return;
      }

      var item = resp.data.results[0];
      if (item && item.lat && item.lng) {
        setLatLng(item.lat, item.lng);
      }
    });
  }

  function initGooglePlaces() {
    var $search = $(fieldSelector('lokalisa_address_search'));
    if (!$search.length) return;

    var input = $search.get(0);
    var autocomplete = new google.maps.places.Autocomplete(input, { fields: ['geometry', 'formatted_address', 'address_components'] });

    autocomplete.addListener('place_changed', function () {
      var place = autocomplete.getPlace();
      if (!place || !place.geometry) return;
      if (place.formatted_address) {
        $search.val(place.formatted_address);
      }
      setLatLng(place.geometry.location.lat(), place.geometry.location.lng());

      var components = place.address_components || [];
      var byType = {};
      components.forEach(function (c) {
        if (c.types && c.types.length) {
          byType[c.types[0]] = c;
        }
      });

      var streetNumber = byType.street_number ? byType.street_number.long_name : '';
      var route = byType.route ? byType.route.long_name : '';
      var city = (byType.locality && byType.locality.long_name) || (byType.administrative_area_level_2 && byType.administrative_area_level_2.long_name) || '';
      var state = byType.administrative_area_level_1 ? byType.administrative_area_level_1.short_name : '';
      var postcode = byType.postal_code ? byType.postal_code.long_name : '';
      var country = byType.country ? byType.country.short_name : '';

      fillAddressFields({
        address1: [route, streetNumber].filter(Boolean).join(' '),
        city: city,
        state: state,
        postcode: postcode,
        country: country
      });
    });
  }

  function googleGeocodeAddress(query) {
    if (!query || query.length < 3) return;
    if (query === lastSearchQuery) return;
    lastSearchQuery = query;

    var geocoder = new google.maps.Geocoder();
    geocoder.geocode({ address: query }, function (results, status) {
      if (status !== 'OK' || !results || !results.length) {
        return;
      }

      var loc = results[0].geometry.location;
      setLatLng(loc.lat(), loc.lng());
    });
  }

  function bindAddressFieldWatcher() {
    var $fields = $(
      fieldSelector('shipping_address_1') + ', ' +
      fieldSelector('shipping_address_2') + ', ' +
      fieldSelector('shipping_city') + ', ' +
      fieldSelector('shipping_state') + ', ' +
      fieldSelector('shipping_postcode') + ', ' +
      fieldSelector('shipping_country')
    );

    if (!$fields.length) return;

    var debounce = null;
    $fields.on('change', function () {
      if (debounce) clearTimeout(debounce);
      debounce = setTimeout(function () {
        var query = buildAddressQuery();
        if (!query) return;

        if (lokalisaCheckout && lokalisaCheckout.provider === 'google') {
          if (window.google && google.maps) {
            googleGeocodeAddress(query);
          }
        } else {
          osmGeocodeAddress(query);
        }
      }, 400);
    });
  }

  function waitForGooglePlaces() {
    var attempts = 0;
    var timer = setInterval(function () {
      attempts += 1;
      if (window.google && google.maps && google.maps.places) {
        clearInterval(timer);
        initGooglePlaces();
        initGoogleMap();
        bindAddressFieldWatcher();
      }
      if (attempts > 20) {
        clearInterval(timer);
      }
    }, 250);
  }

  function initProvider() {
    if (lokalisaCheckout && lokalisaCheckout.provider === 'google') {
      waitForGooglePlaces();
    } else {
      initOsmSearch();
      initLeafletMap();
      bindAddressFieldWatcher();
    }
  }

  $(document).ready(function () {
    initDatepicker();
    initProvider();
  });
})(jQuery);
